﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using Microsoft.Win32;
using Winfoot_7.Controller;
using System.Windows.Forms;
using Winfoot_7_SDK;
using System.Reflection;
using System.IO;
using System.Xml.Serialization;
using System.Xml;
using OpenFileDialog = System.Windows.Forms.OpenFileDialog;
using SaveFileDialog = System.Windows.Forms.SaveFileDialog;

namespace Winfoot_7.Class
{
    public interface IStrategy
    {
        void Pushed();
    }

    public class PluginShowStrategy : IStrategy
    {
        private Control Owner;
        public PluginShowStrategy(Control owner, IControl plugincontrol)
        {
            PluginControl = plugincontrol;
            Owner = owner;
        }
        public IControl PluginControl
        {
            get;
            set;
        }

        #region IStrategy メンバ

        public void Pushed()
        {
            ((Control)PluginControl).Parent = Owner;
            ((Control)PluginControl).Dock = DockStyle.Fill;
            ((Control)PluginControl).Show();
        }

        #endregion

    }

    public class ApplyStrategy : IStrategy
    {

        public ApplyStrategy(ControllerManager controllerManager)
        {
            ControllerManager = controllerManager;
        }

        public ControllerManager ControllerManager
        {
            get;
            set;
        }

        #region IStrategy メンバ

        public void Pushed()
        {
            //適用
            if (MessageBox.Show("適用します。いいですか？","適用確認",MessageBoxButtons.OKCancel,MessageBoxIcon.Information)== DialogResult.OK)
            {
                for (int count = 0; count < ControllerManager.PluginController.ControlList.Count; count++)
                {
                    ControllerManager.PluginController.Apply();
                }
            }
        }

        #endregion
    }

    public class BackUpStrategy : IStrategy
    {
        public BackUpStrategy(PluginController plugincontroller)
        {
            pluginController = plugincontroller;
        }

        public PluginController pluginController
        {
            get;
            set;
        }

        #region IStrategy メンバ

        public void Pushed()
        {
            SaveFileDialog saveFileDialog = new SaveFileDialog();
            saveFileDialog.Filter = "XMLファイル|*.xml";
            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                List<WFRegistry> wfRegistries = pluginController.BackUp();
                XmlTextWriter xmlTextWriter = new XmlTextWriter(saveFileDialog.FileName, Encoding.UTF8);
                xmlTextWriter.Formatting = Formatting.Indented;
                XmlSerializer xmlSerializer = new XmlSerializer(typeof(List<WFRegistry>));
                xmlSerializer.Serialize(xmlTextWriter, wfRegistries);
                xmlTextWriter.Close();
            }

        }

        #endregion
    }

    public class RestorationStrategy : IStrategy
    {
        private interface IRestorationStrategy
        {
            bool IsCorrespondence(WFRegistry wfRegistry);
            void ValueWrite(WFRegistry wfRegistry);
            void ValueDel(WFRegistry wfRegistry);
            void KeyWrite(WFRegistry wfRegistry);
            void KeyDel(WFRegistry wfRegistry);
        }

        public class CLASSES_ROOT_WriterStrategy : IRestorationStrategy
        {

            #region IStrategy メンバ

            public bool IsCorrespondence(WFRegistry wfRegistry)
            {
                return Regex.IsMatch(wfRegistry.Path, @"^HKEY_CLASSES_ROOT\\");
            }

            public void ValueWrite(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CLASSES_ROOT\\", ""));
                registryKey.SetValue(wfRegistry.Name, wfRegistry.Value, wfRegistry.RegistryValueKind);
                registryKey.Close();
            }

            public void ValueDel(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CLASSES_ROOT\\", ""));
                registryKey.DeleteValue(wfRegistry.Name,false);
                registryKey.Close();
            }

            public void KeyWrite(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CLASSES_ROOT\\", ""));
            }

            public void KeyDel(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.DeleteSubKeyTree(Regex.Replace(wfRegistry.Path, @"^HKEY_CLASSES_ROOT\\", ""));
            }

            #endregion
        }

        public class CURRENT_USERWriterStrategy : IRestorationStrategy
        {

            #region IStrategy メンバ

            public bool IsCorrespondence(WFRegistry wfRegistry)
            {
                return Regex.IsMatch(wfRegistry.Path, @"^HKEY_CURRENT_USER\\");
            }

            public void ValueWrite(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CURRENT_USER\\", ""));
                registryKey.SetValue(wfRegistry.Name, wfRegistry.Value, wfRegistry.RegistryValueKind);
                registryKey.Close();
            }

            public void ValueDel(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CURRENT_USER\\", ""));
                registryKey.DeleteValue(wfRegistry.Name, false);
                registryKey.Close();
            }

            public void KeyWrite(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CURRENT_USER\\", ""));
            }

            public void KeyDel(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.DeleteSubKeyTree(Regex.Replace(wfRegistry.Path, @"^HKEY_CURRENT_USER\\", ""));
            }

            #endregion
        }

        public class LOCAL_MACHINE_WriterStrategy : IRestorationStrategy
        {

            #region IStrategy メンバ

            public bool IsCorrespondence(WFRegistry wfRegistry)
            {
                return Regex.IsMatch(wfRegistry.Path, @"^HKEY_LOCAL_MACHINE\\");
            }

            public void ValueWrite(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_LOCAL_MACHINE\\", ""));
                registryKey.SetValue(wfRegistry.Name, wfRegistry.Value, wfRegistry.RegistryValueKind);
                registryKey.Close();
            }

            public void ValueDel(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_LOCAL_MACHINE\\", ""));
                registryKey.DeleteValue(wfRegistry.Name, false);
                registryKey.Close();
            }

            public void KeyWrite(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_LOCAL_MACHINE\\", ""));
            }

            public void KeyDel(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.DeleteSubKeyTree(Regex.Replace(wfRegistry.Path, @"^HKEY_LOCAL_MACHINE\\", ""));
            }

            #endregion
        }

        public class USERS_WriterStrategy : IRestorationStrategy
        {

            #region IStrategy メンバ

            public bool IsCorrespondence(WFRegistry wfRegistry)
            {
                return Regex.IsMatch(wfRegistry.Path, @"^HKEY_USERS\\");
            }

            public void ValueWrite(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_USERS\\", ""));
                registryKey.SetValue(wfRegistry.Name, wfRegistry.Value, wfRegistry.RegistryValueKind);
                registryKey.Close();
            }

            public void ValueDel(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_USERS\\", ""));
                registryKey.DeleteValue(wfRegistry.Name, false);
                registryKey.Close();
            }

            public void KeyWrite(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_USERS\\", ""));
            }

            public void KeyDel(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.DeleteSubKeyTree(Regex.Replace(wfRegistry.Path, @"^HKEY_USERS\\", ""));
            }

            #endregion
        }

        public class CURRENT_CONFIG_WriterStrategy : IRestorationStrategy
        {

            #region IStrategy メンバ

            public bool IsCorrespondence(WFRegistry wfRegistry)
            {
                return Regex.IsMatch(wfRegistry.Path, @"^HKEY_CURRENT_CONFIG\\");
            }

            public void ValueWrite(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CURRENT_CONFIG\\", ""));
                registryKey.SetValue(wfRegistry.Name, wfRegistry.Value, wfRegistry.RegistryValueKind);
                registryKey.Close();
            }

            public void ValueDel(WFRegistry wfRegistry)
            {
                RegistryKey registryKey =
                    Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CURRENT_CONFIG\\", ""));
                registryKey.DeleteValue(wfRegistry.Name, false);
                registryKey.Close();
            }

            public void KeyWrite(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.CreateSubKey(Regex.Replace(wfRegistry.Path, @"^HKEY_CURRENT_CONFIG\\", ""));
            }

            public void KeyDel(WFRegistry wfRegistry)
            {
                Registry.ClassesRoot.DeleteSubKeyTree(Regex.Replace(wfRegistry.Path, @"^HKEY_CURRENT_CONFIG\\", ""));
            }

            #endregion
        }

        private class WFRegistryWriter
        {
            private List<IRestorationStrategy> strategies;
            public WFRegistryWriter()
            {
                strategies = new List<IRestorationStrategy> { new CLASSES_ROOT_WriterStrategy(), new CURRENT_USERWriterStrategy(), new LOCAL_MACHINE_WriterStrategy(), new USERS_WriterStrategy(), new CURRENT_CONFIG_WriterStrategy() };
            }

            public void Write(WFRegistry wfRegistry)
            {
                for (int count = 0; count < strategies.Count; count++)
                {
                    if (strategies[count].IsCorrespondence(wfRegistry))
                    {
                        switch (wfRegistry.RegOperation)
                        {
                            case RegOperation.KeyDel:
                                strategies[count].KeyDel(wfRegistry);
                                break;
                            case RegOperation.KeyWrite:
                                strategies[count].KeyWrite(wfRegistry);
                                break;
                            case RegOperation.ValueDel:
                                strategies[count].ValueDel(wfRegistry);
                                break;
                            case RegOperation.ValueWrite:
                                strategies[count].ValueWrite(wfRegistry);
                                break;
                        }
                    }
                }
            }
        }

        #region IStrategy メンバ

        public void Pushed()
        {
            OpenFileDialog openFileDialog = new OpenFileDialog();
            openFileDialog.Filter = "XMLファイル|*.xml";
            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                XmlSerializer xmlSerializer = new XmlSerializer(typeof(List<WFRegistry>));
                XmlTextReader xmlTextReader = new XmlTextReader(openFileDialog.FileName);
                List<WFRegistry> wfRegistries = (List<WFRegistry>)xmlSerializer.Deserialize(xmlTextReader);
                xmlTextReader.Close();

                WFRegistryWriter wfRegistryWriter = new WFRegistryWriter();
                for (int count = 0; count < wfRegistries.Count; count++)
                {
                    wfRegistryWriter.Write(wfRegistries[count]);
                }
            }
        }

        #endregion
    }

    public class ReturnToRootStrategy : IStrategy
    {
        public ReturnToRootStrategy(ControllerManager controller)
        {
            ControllerManager = controller;
        }

        public ControllerManager ControllerManager
        {
            get;
            set;
        }

        #region IStrategy メンバ

        public void Pushed()
        {
            ControllerManager.PluginController.Hide();
            ControllerManager.MenuController.ReturnToRoot();
        }

        #endregion
    }

    public class ArticleReadStratety : IStrategy
    {
        public ArticleReadStratety(ControllerManager controller, string path, Control pluginOwner)
        {
            ControllerManager = controller;
            Path = path;
            PluginOwner = pluginOwner;
            LoadDLL();
        }

        public Control PluginOwner
        {
            get;
            set;
        }

        public ControllerManager ControllerManager
        {
            get;
            set;
        }

        public string Path
        {
            get;
            set;
        }

        public List<IMenuItem> MenuList
        {
            get;
            set;
        }

        private void LoadDLL()
        {
            if (MenuList == null)
            {
                //プラグインのロードおよびメニューの作成
                List<IMenuItem> menuItems = new List<IMenuItem>();

                //IPlugin型の名前
                string ipluginName = typeof(IPlugin).FullName;

                //.dllファイルを探す
                string[] dlls = System.IO.Directory.GetFiles(Path, "*.dll");

                foreach (string dll in dlls)
                {
                    try
                    {
                        //アセンブリとして読み込む
                        Assembly asm = Assembly.LoadFrom(dll);

                        foreach (Type t in asm.GetTypes())
                        {
                            //アセンブリ内のすべての型について、
                            //プラグインとして有効か調べる
                            if (t.IsClass && t.IsPublic && !t.IsAbstract && t.GetInterface(ipluginName) != null)
                            {
                                //PluginInfoをコレクションに追加す
                                IPlugin plugin = (IPlugin)asm.CreateInstance(t.FullName);
                                ControllerManager.PluginController.ControlList.Add(plugin.GetControl());
                                menuItems.Add(new WinfootMenuItem(plugin.Icon, plugin.PluginName, new PluginShowStrategy(PluginOwner, plugin.GetControl())));
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        //きちんと例外処理する必要あり
                        MessageBox.Show(dll + " : " + e);
                    }
                }


                //戻るボタンの作成
                menuItems.Add(new WinfootMenuItem(Properties.Resources.restart, "戻る", new ReturnToRootStrategy(ControllerManager)));

                MenuList = menuItems;
            }
        }

        #region IStrategy メンバ

        public void Pushed()
        {
            //メニューの表示
            ControllerManager.MenuController.SetMenuItems(MenuList);
        }

        #endregion
    }

    public class WinfootMenuItem : IMenuItem
    {
        public WinfootMenuItem(Image image, string text, IStrategy strategy)
        {
            Image = image;
            Text = text;
            this.strategy = strategy;
        }

        public IStrategy strategy
        {
            get;
            set;
        }

        #region IMenuItem メンバ

        public Image Image
        {
            get;
            set;
        }

        public string Text
        {
            get;
            set;
        }

        public void Pushed()
        {
            strategy.Pushed();
        }

        #endregion
    }

    public interface IMenuItem
    {
        Image Image
        {
            get;
            set;
        }
        string Text
        {
            get;
            set;
        }

        void Pushed();
    }
}